<?php

namespace App\Http\Controllers;

use App\Helpers\Image\Frames\Desktop;
use App\Helpers\Image\Frames\Laptop;
use App\Helpers\Image\Frames\Phone;
use App\Helpers\Image\Frames\Tablet;
use App\Helpers\Image\Showcase;
use App\Helpers\Image\Watermark;
use App\Http\Requests;
use Screen\Capture;

class ShowcaseController extends Controller
{
    public function __construct(Capture $screenCapture)
    {
        $this->screenCapture = $screenCapture;
    }

    public function generateShowcaseImage($optionsBase64)
    {
        $options = json_decode(base64_decode($optionsBase64));

        $showcase = new Showcase();
        $showcase->setTemplate($options->template);

        $desktopFrame = null;
        $laptopFrame = null;
        $tabletFrame = null;
        $phoneFrame = null;

        switch ($options->template) {
            case Showcase::TEMPLATE_DESKTOP:
                $desktopFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_DESKTOP);
                break;
            case Showcase::TEMPLATE_LAPTOP:
                $laptopFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_LAPTOP);
                break;
            case Showcase::TEMPLATE_TABLET:
                $tabletFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_TABLET);
                break;
            case Showcase::TEMPLATE_PHONE:
                $phoneFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_PHONE);
                break;
            case Showcase::TEMPLATE_ALL:
            default:
                $desktopFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_DESKTOP);
                $laptopFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_LAPTOP);
                $tabletFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_TABLET);
                $phoneFrame = $this->getScreenShotFrame($options->url, Showcase::TEMPLATE_PHONE);
                break;
        }

        if ($desktopFrame) {
            $showcase->setDesktopLayer($desktopFrame);
        }
        if ($laptopFrame) {
            $showcase->setLaptopLayer($laptopFrame);
        }
        if ($tabletFrame) {
            $showcase->setTabletLayer($tabletFrame);
        }
        if ($phoneFrame) {
            $showcase->setPhoneLayer($phoneFrame);
        }

        $dirPath = public_path() . "/images/showcase";
        $filename = md5($optionsBase64) . '.png';

        $image = $showcase->getImage();
        $image->save($dirPath, $filename, true, str_replace('#', '', $options->bgColor), 95);

        Watermark::apply($dirPath, $filename);
        
        return response()->json([
            'status'  => true,
            'image'   => [
                'src'    => asset('images/showcase/' . $filename),
                'width'  => $image->getWidth(),
                'height' => $image->getHeight(),
            ],
            'options' => $options,
        ]);

        //header('Content-type: image/png');
        //imagepng($showcase->getImage()->getResult('8dceeb'));

        // return image url by json
    }

    /**
     * @param        $url
     * @param string $type
     * @return Frame
     * @throws \Exception
     */
    protected function getScreenShotFrame($url, $type = 'desktop')
    {
        $outputName = sprintf("screen-%s-%s.jpg", md5($url), $type);
        $outputFilePath = $this->screenCapture->output->getLocation() . $outputName;

        switch ($type) {
            case 'phone':
                $this->screenCapture->setUserAgentString("Mozilla/5.0 (iPhone; CPU iPhone OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9A334 Safari/7534.48.3");
                $this->screenCapture->setWidth(320);
                $this->screenCapture->setHeight(568);
                $this->screenCapture->setClipWidth(320);
                $this->screenCapture->setClipHeight(568);
                $frame = new Phone($outputFilePath);
                break;

            case 'tablet':
                $this->screenCapture->setUserAgentString("Mozilla/5.0 (iPad; CPU OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/5.1 Mobile/9A334 Safari/7534.48.3");
                $this->screenCapture->setWidth(768);
                $this->screenCapture->setHeight(1024);
                $this->screenCapture->setClipWidth(768);
                $this->screenCapture->setClipHeight(1024);
                $frame = new Tablet($outputFilePath);
                break;

            case 'laptop':
                $this->screenCapture->setUserAgentString("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.75.14 (KHTML, like Gecko) Version/7.0.3 Safari/7046A194A");
                $this->screenCapture->setWidth(1024);
                $this->screenCapture->setHeight(768);
                $this->screenCapture->setClipWidth(1024);
                $this->screenCapture->setClipHeight(768);
                $frame = new Laptop($outputFilePath);
                break;

            case 'desktop':
            default:
                $this->screenCapture->setUserAgentString("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.75.14 (KHTML, like Gecko) Version/7.0.3 Safari/7046A194A");
                $this->screenCapture->setWidth(1280);
                $this->screenCapture->setHeight(800);
                $this->screenCapture->setClipWidth(1280);
                $this->screenCapture->setClipHeight(800);
                $frame = new Desktop($outputFilePath);
                break;
        }

        $this->screenCapture->setUrl($url);
        $this->screenCapture->save($outputName);

        return $frame;
    }
}
